"""
╔═══════════════════════════════════════════════════════════════════════════════╗
║              SCOREINVEST PORTFOLIO TRACKER v2.0 - PREMIUM                    ║
║                 Suivez votre patrimoine comme un pro                          ║
║                          © 2026 ScoreInvest.fr                               ║
╚═══════════════════════════════════════════════════════════════════════════════╝

Nouveautés v2.0:
- 📈 Actualisation automatique des cours (Yahoo Finance + CoinGecko)
- 🔔 Alertes de prix personnalisables
- 📊 Analyse avancée (performance, allocation, dividendes)
- 📅 Historique des transactions
- 🎯 Allocation cible avec rééquilibrage
- 💰 Suivi des dividendes
- 📱 Watchlist
- 🌓 Thèmes clair/sombre
- 💾 Backup/Restore
- 📄 Export PDF des rapports
"""

import sys
import sqlite3
import json
import csv
import threading
import urllib.request
import urllib.error
from datetime import datetime, timedelta
from pathlib import Path
from typing import Optional, Dict, List
from dataclasses import dataclass

from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QPushButton, QLineEdit, QComboBox, QTableWidget, QTableWidgetItem,
    QFrame, QStackedWidget, QFileDialog, QMessageBox, QDialog, QFormLayout,
    QDoubleSpinBox, QDateEdit, QHeaderView, QScrollArea, QGridLayout,
    QSpinBox, QProgressBar, QTabWidget, QCheckBox, QSlider, QSplitter,
    QListWidget, QListWidgetItem, QMenu, QSystemTrayIcon, QTextEdit,
    QGroupBox, QRadioButton, QButtonGroup, QToolButton, QSizePolicy
)
from PyQt6.QtCore import (
    Qt, QDate, QTimer, QThread, pyqtSignal, QSize, QPropertyAnimation,
    QEasingCurve, QDateTime, QPoint, QMargins
)
from PyQt6.QtGui import (
    QFont, QColor, QPalette, QIcon, QPainter, QBrush, QPen, QAction,
    QPixmap, QLinearGradient, QCursor
)
from PyQt6.QtCharts import (
    QChart, QChartView, QPieSeries, QLineSeries, QDateTimeAxis, 
    QValueAxis, QBarSeries, QBarSet, QBarCategoryAxis, QAreaSeries
)

# ═══════════════════════════════════════════════════════════════════════════════
# 🎨 THÈMES
# ═══════════════════════════════════════════════════════════════════════════════

THEMES = {
    'dark': {
        'name': 'Sombre',
        'bg_primary': '#0a0a0f',
        'bg_secondary': '#12121a',
        'bg_tertiary': '#1a1a25',
        'bg_hover': '#252535',
        'border': '#2a2a3a',
        'border_light': '#3a3a4a',
        'text_primary': '#f0f0f5',
        'text_secondary': '#8888a0',
        'text_muted': '#5a5a6a',
        'accent': '#d4af37',
        'accent_hover': '#e6c349',
        'success': '#10b981',
        'danger': '#ef4444',
        'warning': '#f59e0b',
        'info': '#3b82f6',
        'purple': '#8b5cf6',
        'cyan': '#06b6d4',
    },
    'light': {
        'name': 'Clair',
        'bg_primary': '#f8f9fa',
        'bg_secondary': '#ffffff',
        'bg_tertiary': '#f1f3f4',
        'bg_hover': '#e8eaed',
        'border': '#dadce0',
        'border_light': '#e8eaed',
        'text_primary': '#202124',
        'text_secondary': '#5f6368',
        'text_muted': '#9aa0a6',
        'accent': '#b8860b',
        'accent_hover': '#d4a017',
        'success': '#0d9488',
        'danger': '#dc2626',
        'warning': '#d97706',
        'info': '#2563eb',
        'purple': '#7c3aed',
        'cyan': '#0891b2',
    }
}

def get_stylesheet(theme: dict) -> str:
    return f"""
    /* ═══════════════════════════════════════════════════════════════════════════
       SCOREINVEST PORTFOLIO TRACKER v2.0 - DYNAMIC THEME
       ═══════════════════════════════════════════════════════════════════════════ */
    
    * {{ font-family: 'Segoe UI', 'SF Pro Display', -apple-system, sans-serif; }}
    
    QMainWindow, QWidget {{ background-color: {theme['bg_primary']}; color: {theme['text_primary']}; }}
    
    /* SIDEBAR */
    #sidebar {{ background-color: {theme['bg_secondary']}; border-right: 1px solid {theme['border']}; }}
    
    #sidebar QPushButton {{
        background-color: transparent;
        color: {theme['text_secondary']};
        border: none;
        border-radius: 10px;
        padding: 14px 18px;
        text-align: left;
        font-size: 13px;
        font-weight: 500;
        margin: 3px 10px;
    }}
    
    #sidebar QPushButton:hover {{ background-color: {theme['bg_tertiary']}; color: {theme['text_primary']}; }}
    
    #sidebar QPushButton:checked, #sidebar QPushButton[active="true"] {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:0, 
            stop:0 {theme['accent']}30, stop:1 transparent);
        color: {theme['accent']};
        border-left: 3px solid {theme['accent']};
        margin-left: 7px;
    }}
    
    #logo {{ color: {theme['accent']}; font-size: 20px; font-weight: 800; padding: 20px 18px; }}
    
    /* CARDS */
    QFrame[class="card"] {{
        background-color: {theme['bg_secondary']};
        border: 1px solid {theme['border']};
        border-radius: 14px;
    }}
    
    QFrame[class="card"]:hover {{ border-color: {theme['border_light']}; }}
    
    QFrame[class="stat-card"] {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:1, 
            stop:0 {theme['bg_secondary']}, stop:1 {theme['bg_tertiary']});
        border: 1px solid {theme['border']};
        border-radius: 14px;
    }}
    
    QFrame[class="stat-card-accent"] {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:1, 
            stop:0 {theme['accent']}15, stop:1 {theme['bg_secondary']});
        border: 1px solid {theme['accent']}40;
        border-radius: 14px;
    }}
    
    /* LABELS */
    QLabel {{ color: {theme['text_primary']}; }}
    QLabel[class="title"] {{ font-size: 26px; font-weight: 800; }}
    QLabel[class="subtitle"] {{ font-size: 14px; color: {theme['text_secondary']}; }}
    QLabel[class="stat-value"] {{ font-size: 28px; font-weight: 800; color: {theme['accent']}; }}
    QLabel[class="stat-value-success"] {{ font-size: 28px; font-weight: 800; color: {theme['success']}; }}
    QLabel[class="stat-value-danger"] {{ font-size: 28px; font-weight: 800; color: {theme['danger']}; }}
    QLabel[class="stat-label"] {{ font-size: 12px; color: {theme['text_secondary']}; font-weight: 500; }}
    QLabel[class="section-title"] {{ font-size: 16px; font-weight: 700; padding: 8px 0; }}
    QLabel[class="small"] {{ font-size: 11px; color: {theme['text_muted']}; }}
    
    /* BUTTONS */
    QPushButton {{
        background-color: {theme['bg_tertiary']};
        color: {theme['text_primary']};
        border: 1px solid {theme['border']};
        border-radius: 8px;
        padding: 10px 20px;
        font-size: 13px;
        font-weight: 600;
    }}
    
    QPushButton:hover {{ background-color: {theme['bg_hover']}; border-color: {theme['border_light']}; }}
    QPushButton:pressed {{ background-color: {theme['bg_tertiary']}; }}
    
    QPushButton[class="primary"] {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:0, 
            stop:0 {theme['accent']}, stop:1 {theme['accent_hover']});
        color: {theme['bg_primary']};
        border: none;
        font-weight: 700;
    }}
    
    QPushButton[class="primary"]:hover {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:0, 
            stop:0 {theme['accent_hover']}, stop:1 {theme['accent']});
    }}
    
    QPushButton[class="success"] {{ background-color: {theme['success']}; color: white; border: none; }}
    QPushButton[class="danger"] {{ background-color: {theme['danger']}; color: white; border: none; }}
    QPushButton[class="ghost"] {{ background-color: transparent; border: 1px solid {theme['border']}; }}
    QPushButton[class="ghost"]:hover {{ background-color: {theme['bg_tertiary']}; }}
    
    QPushButton[class="icon-btn"] {{
        background: transparent;
        border: none;
        border-radius: 6px;
        padding: 6px;
        min-width: 32px;
        min-height: 32px;
    }}
    
    QPushButton[class="icon-btn"]:hover {{ background-color: {theme['bg_tertiary']}; }}
    
    /* INPUTS */
    QLineEdit, QDoubleSpinBox, QSpinBox, QComboBox, QDateEdit, QTextEdit {{
        background-color: {theme['bg_tertiary']};
        color: {theme['text_primary']};
        border: 1px solid {theme['border']};
        border-radius: 8px;
        padding: 10px 14px;
        font-size: 13px;
    }}
    
    QLineEdit:focus, QDoubleSpinBox:focus, QSpinBox:focus, QComboBox:focus, QDateEdit:focus, QTextEdit:focus {{
        border-color: {theme['accent']};
        background-color: {theme['bg_hover']};
    }}
    
    QComboBox::drop-down {{ border: none; padding-right: 12px; }}
    QComboBox::down-arrow {{ border-left: 5px solid transparent; border-right: 5px solid transparent; border-top: 5px solid {theme['text_secondary']}; }}
    
    QComboBox QAbstractItemView {{
        background-color: {theme['bg_secondary']};
        color: {theme['text_primary']};
        border: 1px solid {theme['border']};
        selection-background-color: {theme['accent']};
        selection-color: {theme['bg_primary']};
    }}
    
    /* TABLE */
    QTableWidget {{
        background-color: {theme['bg_secondary']};
        color: {theme['text_primary']};
        border: 1px solid {theme['border']};
        border-radius: 10px;
        gridline-color: {theme['border']};
    }}
    
    QTableWidget::item {{ padding: 10px; border-bottom: 1px solid {theme['bg_tertiary']}; }}
    QTableWidget::item:selected {{ background-color: {theme['accent']}30; color: {theme['text_primary']}; }}
    QTableWidget::item:hover {{ background-color: {theme['bg_tertiary']}; }}
    
    QHeaderView::section {{
        background-color: {theme['bg_primary']};
        color: {theme['text_secondary']};
        padding: 12px 10px;
        border: none;
        border-bottom: 1px solid {theme['border']};
        font-weight: 600;
        font-size: 11px;
        text-transform: uppercase;
    }}
    
    /* TABS */
    QTabWidget::pane {{ border: 1px solid {theme['border']}; border-radius: 10px; background: {theme['bg_secondary']}; }}
    
    QTabBar::tab {{
        background-color: {theme['bg_tertiary']};
        color: {theme['text_secondary']};
        border: 1px solid {theme['border']};
        border-bottom: none;
        border-radius: 8px 8px 0 0;
        padding: 10px 20px;
        margin-right: 2px;
        font-weight: 500;
    }}
    
    QTabBar::tab:selected {{ background-color: {theme['bg_secondary']}; color: {theme['accent']}; border-bottom: 2px solid {theme['accent']}; }}
    QTabBar::tab:hover {{ background-color: {theme['bg_hover']}; }}
    
    /* SCROLLBAR */
    QScrollBar:vertical {{
        background-color: {theme['bg_secondary']};
        width: 8px;
        border-radius: 4px;
    }}
    
    QScrollBar::handle:vertical {{
        background-color: {theme['border']};
        border-radius: 4px;
        min-height: 30px;
    }}
    
    QScrollBar::handle:vertical:hover {{ background-color: {theme['border_light']}; }}
    QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {{ height: 0; }}
    
    /* PROGRESS BAR */
    QProgressBar {{
        background-color: {theme['bg_tertiary']};
        border: none;
        border-radius: 6px;
        height: 12px;
        text-align: center;
        color: {theme['text_primary']};
        font-weight: 600;
        font-size: 10px;
    }}
    
    QProgressBar::chunk {{
        background: qlineargradient(x1:0, y1:0, x2:1, y2:0, 
            stop:0 {theme['accent']}, stop:1 {theme['accent_hover']});
        border-radius: 6px;
    }}
    
    /* CHECKBOX */
    QCheckBox {{ color: {theme['text_primary']}; spacing: 8px; }}
    QCheckBox::indicator {{ width: 18px; height: 18px; border-radius: 4px; border: 2px solid {theme['border']}; }}
    QCheckBox::indicator:checked {{ background-color: {theme['accent']}; border-color: {theme['accent']}; }}
    
    /* GROUP BOX */
    QGroupBox {{
        font-weight: 600;
        border: 1px solid {theme['border']};
        border-radius: 10px;
        margin-top: 10px;
        padding-top: 15px;
    }}
    
    QGroupBox::title {{ subcontrol-origin: margin; left: 15px; padding: 0 8px; color: {theme['text_secondary']}; }}
    
    /* SLIDER */
    QSlider::groove:horizontal {{ background: {theme['bg_tertiary']}; height: 6px; border-radius: 3px; }}
    QSlider::handle:horizontal {{ background: {theme['accent']}; width: 16px; height: 16px; margin: -5px 0; border-radius: 8px; }}
    QSlider::sub-page:horizontal {{ background: {theme['accent']}; border-radius: 3px; }}
    
    /* DIALOG */
    QDialog {{ background-color: {theme['bg_secondary']}; border: 1px solid {theme['border']}; border-radius: 14px; }}
    QMessageBox {{ background-color: {theme['bg_secondary']}; }}
    QMessageBox QLabel {{ color: {theme['text_primary']}; }}
    
    /* TOOLTIP */
    QToolTip {{
        background-color: {theme['bg_tertiary']};
        color: {theme['text_primary']};
        border: 1px solid {theme['border']};
        border-radius: 6px;
        padding: 6px 10px;
        font-size: 12px;
    }}
    
    /* LIST WIDGET */
    QListWidget {{
        background-color: {theme['bg_secondary']};
        border: 1px solid {theme['border']};
        border-radius: 8px;
    }}
    
    QListWidget::item {{ padding: 10px; border-bottom: 1px solid {theme['bg_tertiary']}; }}
    QListWidget::item:selected {{ background-color: {theme['accent']}30; }}
    QListWidget::item:hover {{ background-color: {theme['bg_tertiary']}; }}
    """


# ═══════════════════════════════════════════════════════════════════════════════
# 🗄️ DATABASE v2
# ═══════════════════════════════════════════════════════════════════════════════

class Database:
    def __init__(self):
        self.db_path = Path.home() / ".scoreinvest_tracker" / "portfolio_v2.db"
        self.db_path.parent.mkdir(parents=True, exist_ok=True)
        self.conn = sqlite3.connect(str(self.db_path), check_same_thread=False)
        self.conn.row_factory = sqlite3.Row
        self.init_tables()
    
    def init_tables(self):
        cursor = self.conn.cursor()
        
        # Actifs
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS assets (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                ticker TEXT,
                isin TEXT,
                category TEXT NOT NULL,
                subcategory TEXT,
                quantity REAL NOT NULL,
                buy_price REAL NOT NULL,
                current_price REAL,
                currency TEXT DEFAULT 'EUR',
                account TEXT,
                broker TEXT,
                notes TEXT,
                dividend_yield REAL DEFAULT 0,
                last_dividend_date TEXT,
                auto_update INTEGER DEFAULT 1,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                updated_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # Transactions
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS transactions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                asset_id INTEGER,
                type TEXT NOT NULL,
                quantity REAL NOT NULL,
                price REAL NOT NULL,
                fees REAL DEFAULT 0,
                date TEXT NOT NULL,
                notes TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (asset_id) REFERENCES assets(id) ON DELETE CASCADE
            )
        """)
        
        # Dividendes
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS dividends (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                asset_id INTEGER,
                amount REAL NOT NULL,
                date TEXT NOT NULL,
                type TEXT DEFAULT 'dividend',
                withholding_tax REAL DEFAULT 0,
                notes TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (asset_id) REFERENCES assets(id) ON DELETE CASCADE
            )
        """)
        
        # Historique portfolio
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS portfolio_history (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL UNIQUE,
                total_value REAL NOT NULL,
                total_invested REAL NOT NULL,
                total_dividends REAL DEFAULT 0,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # Objectifs
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS goals (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                target_amount REAL NOT NULL,
                target_date TEXT,
                monthly_contribution REAL DEFAULT 0,
                priority INTEGER DEFAULT 2,
                color TEXT DEFAULT '#d4af37',
                created_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # Allocation cible
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS target_allocation (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                category TEXT NOT NULL UNIQUE,
                target_percent REAL NOT NULL,
                color TEXT
            )
        """)
        
        # Alertes
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS alerts (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                asset_id INTEGER,
                type TEXT NOT NULL,
                condition TEXT NOT NULL,
                value REAL NOT NULL,
                triggered INTEGER DEFAULT 0,
                enabled INTEGER DEFAULT 1,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (asset_id) REFERENCES assets(id) ON DELETE CASCADE
            )
        """)
        
        # Watchlist
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS watchlist (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                ticker TEXT,
                category TEXT,
                target_price REAL,
                notes TEXT,
                current_price REAL,
                last_update TEXT,
                created_at TEXT DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # Paramètres
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS settings (
                key TEXT PRIMARY KEY,
                value TEXT
            )
        """)
        
        # Valeurs par défaut allocation
        default_allocation = [
            ('Actions', 40, '#3b82f6'),
            ('ETF', 30, '#d4af37'),
            ('Obligations', 10, '#10b981'),
            ('Crypto', 10, '#f59e0b'),
            ('Immobilier', 5, '#8b5cf6'),
            ('Liquidités', 5, '#6b7280'),
        ]
        
        for cat, pct, color in default_allocation:
            cursor.execute("""
                INSERT OR IGNORE INTO target_allocation (category, target_percent, color)
                VALUES (?, ?, ?)
            """, (cat, pct, color))
        
        self.conn.commit()
    
    # === ASSETS ===
    def add_asset(self, **kwargs):
        cursor = self.conn.cursor()
        columns = ', '.join(kwargs.keys())
        placeholders = ', '.join(['?' for _ in kwargs])
        cursor.execute(f"INSERT INTO assets ({columns}) VALUES ({placeholders})", list(kwargs.values()))
        self.conn.commit()
        return cursor.lastrowid
    
    def update_asset(self, asset_id, **kwargs):
        cursor = self.conn.cursor()
        updates = ', '.join([f"{k} = ?" for k in kwargs.keys()])
        values = list(kwargs.values()) + [asset_id]
        cursor.execute(f"UPDATE assets SET {updates}, updated_at = CURRENT_TIMESTAMP WHERE id = ?", values)
        self.conn.commit()
    
    def delete_asset(self, asset_id):
        cursor = self.conn.cursor()
        cursor.execute("DELETE FROM assets WHERE id = ?", (asset_id,))
        self.conn.commit()
    
    def get_assets(self):
        cursor = self.conn.cursor()
        cursor.execute("SELECT * FROM assets ORDER BY category, name")
        return [dict(row) for row in cursor.fetchall()]
    
    def get_asset(self, asset_id):
        cursor = self.conn.cursor()
        cursor.execute("SELECT * FROM assets WHERE id = ?", (asset_id,))
        row = cursor.fetchone()
        return dict(row) if row else None
    
    def update_asset_price(self, asset_id, price):
        cursor = self.conn.cursor()
        cursor.execute("UPDATE assets SET current_price = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?", (price, asset_id))
        self.conn.commit()
    
    # === TRANSACTIONS ===
    def add_transaction(self, asset_id, type_, quantity, price, fees=0, date=None, notes=""):
        cursor = self.conn.cursor()
        date = date or datetime.now().strftime("%Y-%m-%d")
        cursor.execute("""
            INSERT INTO transactions (asset_id, type, quantity, price, fees, date, notes)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        """, (asset_id, type_, quantity, price, fees, date, notes))
        self.conn.commit()
        return cursor.lastrowid
    
    def get_transactions(self, asset_id=None, limit=50):
        cursor = self.conn.cursor()
        if asset_id:
            cursor.execute("""
                SELECT t.*, a.name as asset_name, a.ticker 
                FROM transactions t 
                JOIN assets a ON t.asset_id = a.id 
                WHERE t.asset_id = ?
                ORDER BY t.date DESC LIMIT ?
            """, (asset_id, limit))
        else:
            cursor.execute("""
                SELECT t.*, a.name as asset_name, a.ticker 
                FROM transactions t 
                JOIN assets a ON t.asset_id = a.id 
                ORDER BY t.date DESC LIMIT ?
            """, (limit,))
        return [dict(row) for row in cursor.fetchall()]
    
    # === DIVIDENDES ===
    def add_dividend(self, asset_id, amount, date, type_='dividend', withholding_tax=0, notes=""):
        cursor = self.conn.cursor()
        cursor.execute("""
            INSERT INTO dividends (asset_id, amount, date, type, withholding_tax, notes)
            VALUES (?, ?, ?, ?, ?, ?)
        """, (asset_id, amount, date, type_, withholding_tax, notes))
        self.conn.commit()
        return cursor.lastrowid
    
    def get_dividends(self, asset_id=None, year=None):
        cursor = self.conn.cursor()
        query = "SELECT d.*, a.name as asset_name FROM dividends d JOIN assets a ON d.asset_id = a.id"
        params = []
        conditions = []
        
        if asset_id:
            conditions.append("d.asset_id = ?")
            params.append(asset_id)
        if year:
            conditions.append("strftime('%Y', d.date) = ?")
            params.append(str(year))
        
        if conditions:
            query += " WHERE " + " AND ".join(conditions)
        query += " ORDER BY d.date DESC"
        
        cursor.execute(query, params)
        return [dict(row) for row in cursor.fetchall()]
    
    def get_total_dividends(self, year=None):
        cursor = self.conn.cursor()
        if year:
            cursor.execute("""
                SELECT COALESCE(SUM(amount - withholding_tax), 0) as total 
                FROM dividends WHERE strftime('%Y', date) = ?
            """, (str(year),))
        else:
            cursor.execute("SELECT COALESCE(SUM(amount - withholding_tax), 0) as total FROM dividends")
        return cursor.fetchone()['total']
    
    # === HISTORIQUE ===
    def save_snapshot(self, total_value, total_invested, total_dividends=0):
        cursor = self.conn.cursor()
        today = datetime.now().strftime("%Y-%m-%d")
        cursor.execute("""
            INSERT OR REPLACE INTO portfolio_history (date, total_value, total_invested, total_dividends)
            VALUES (?, ?, ?, ?)
        """, (today, total_value, total_invested, total_dividends))
        self.conn.commit()
    
    def get_history(self, days=365):
        cursor = self.conn.cursor()
        cursor.execute("""
            SELECT * FROM portfolio_history 
            WHERE date >= date('now', ?) ORDER BY date
        """, (f'-{days} days',))
        return [dict(row) for row in cursor.fetchall()]
    
    # === OBJECTIFS ===
    def add_goal(self, name, target_amount, target_date=None, monthly_contribution=0, priority=2, color='#d4af37'):
        cursor = self.conn.cursor()
        cursor.execute("""
            INSERT INTO goals (name, target_amount, target_date, monthly_contribution, priority, color)
            VALUES (?, ?, ?, ?, ?, ?)
        """, (name, target_amount, target_date, monthly_contribution, priority, color))
        self.conn.commit()
        return cursor.lastrowid
    
    def get_goals(self):
        cursor = self.conn.cursor()
        cursor.execute("SELECT * FROM goals ORDER BY priority, created_at")
        return [dict(row) for row in cursor.fetchall()]
    
    def delete_goal(self, goal_id):
        cursor = self.conn.cursor()
        cursor.execute("DELETE FROM goals WHERE id = ?", (goal_id,))
        self.conn.commit()
    
    # === ALLOCATION ===
    def get_target_allocation(self):
        cursor = self.conn.cursor()
        cursor.execute("SELECT * FROM target_allocation ORDER BY target_percent DESC")
        return [dict(row) for row in cursor.fetchall()]
    
    def set_target_allocation(self, category, percent, color=None):
        cursor = self.conn.cursor()
        if color:
            cursor.execute("""
                INSERT OR REPLACE INTO target_allocation (category, target_percent, color)
                VALUES (?, ?, ?)
            """, (category, percent, color))
        else:
            cursor.execute("""
                UPDATE target_allocation SET target_percent = ? WHERE category = ?
            """, (percent, category))
        self.conn.commit()
    
    # === ALERTES ===
    def add_alert(self, asset_id, type_, condition, value):
        cursor = self.conn.cursor()
        cursor.execute("""
            INSERT INTO alerts (asset_id, type, condition, value)
            VALUES (?, ?, ?, ?)
        """, (asset_id, type_, condition, value))
        self.conn.commit()
        return cursor.lastrowid
    
    def get_alerts(self, enabled_only=True):
        cursor = self.conn.cursor()
        query = """
            SELECT al.*, a.name as asset_name, a.current_price
            FROM alerts al
            JOIN assets a ON al.asset_id = a.id
        """
        if enabled_only:
            query += " WHERE al.enabled = 1"
        cursor.execute(query)
        return [dict(row) for row in cursor.fetchall()]
    
    def delete_alert(self, alert_id):
        cursor = self.conn.cursor()
        cursor.execute("DELETE FROM alerts WHERE id = ?", (alert_id,))
        self.conn.commit()
    
    # === WATCHLIST ===
    def add_to_watchlist(self, name, ticker=None, category=None, target_price=None, notes=""):
        cursor = self.conn.cursor()
        cursor.execute("""
            INSERT INTO watchlist (name, ticker, category, target_price, notes)
            VALUES (?, ?, ?, ?, ?)
        """, (name, ticker, category, target_price, notes))
        self.conn.commit()
        return cursor.lastrowid
    
    def get_watchlist(self):
        cursor = self.conn.cursor()
        cursor.execute("SELECT * FROM watchlist ORDER BY created_at DESC")
        return [dict(row) for row in cursor.fetchall()]
    
    def remove_from_watchlist(self, item_id):
        cursor = self.conn.cursor()
        cursor.execute("DELETE FROM watchlist WHERE id = ?", (item_id,))
        self.conn.commit()
    
    # === SETTINGS ===
    def get_setting(self, key, default=None):
        cursor = self.conn.cursor()
        cursor.execute("SELECT value FROM settings WHERE key = ?", (key,))
        row = cursor.fetchone()
        return row['value'] if row else default
    
    def set_setting(self, key, value):
        cursor = self.conn.cursor()
        cursor.execute("INSERT OR REPLACE INTO settings (key, value) VALUES (?, ?)", (key, str(value)))
        self.conn.commit()
    
    # === STATS ===
    def get_portfolio_stats(self):
        assets = self.get_assets()
        total_invested = sum(a['quantity'] * a['buy_price'] for a in assets)
        total_value = sum(a['quantity'] * (a['current_price'] or a['buy_price']) for a in assets)
        total_dividends = self.get_total_dividends()
        gain = total_value - total_invested
        gain_pct = (gain / total_invested * 100) if total_invested > 0 else 0
        
        # Par catégorie
        categories = {}
        for a in assets:
            cat = a['category']
            value = a['quantity'] * (a['current_price'] or a['buy_price'])
            categories[cat] = categories.get(cat, 0) + value
        
        return {
            'total_invested': total_invested,
            'total_value': total_value,
            'total_dividends': total_dividends,
            'gain': gain,
            'gain_pct': gain_pct,
            'gain_with_dividends': gain + total_dividends,
            'categories': categories,
            'asset_count': len(assets)
        }
    
    # === BACKUP ===
    def export_backup(self, filepath):
        import shutil
        shutil.copy(str(self.db_path), filepath)
    
    def import_backup(self, filepath):
        import shutil
        self.conn.close()
        shutil.copy(filepath, str(self.db_path))
        self.conn = sqlite3.connect(str(self.db_path), check_same_thread=False)
        self.conn.row_factory = sqlite3.Row


# ═══════════════════════════════════════════════════════════════════════════════
# 🌐 API DE PRIX (Yahoo Finance + CoinGecko)
# ═══════════════════════════════════════════════════════════════════════════════

class PriceUpdateThread(QThread):
    """Thread pour mise à jour des prix en arrière-plan"""
    price_updated = pyqtSignal(int, float)  # asset_id, new_price
    update_complete = pyqtSignal()
    error_occurred = pyqtSignal(str)
    
    def __init__(self, assets: List[dict]):
        super().__init__()
        self.assets = assets
    
    def run(self):
        for asset in self.assets:
            if not asset.get('auto_update'):
                continue
            
            ticker = asset.get('ticker', '')
            if not ticker:
                continue
            
            try:
                price = self.fetch_price(ticker, asset.get('category', ''))
                if price and price > 0:
                    self.price_updated.emit(asset['id'], price)
            except Exception as e:
                self.error_occurred.emit(f"Erreur {ticker}: {str(e)}")
        
        self.update_complete.emit()
    
    def fetch_price(self, ticker: str, category: str) -> Optional[float]:
        """Récupère le prix depuis Yahoo Finance ou CoinGecko"""
        
        # Crypto -> CoinGecko
        if category.lower() == 'crypto':
            return self.fetch_crypto_price(ticker)
        
        # Actions/ETF -> Yahoo Finance
        return self.fetch_yahoo_price(ticker)
    
    def fetch_yahoo_price(self, ticker: str) -> Optional[float]:
        """Prix via Yahoo Finance"""
        try:
            # Ajouter .PA pour les tickers français sans suffixe
            if '.' not in ticker and len(ticker) <= 5:
                ticker = f"{ticker}.PA"
            
            url = f"https://query1.finance.yahoo.com/v8/finance/chart/{ticker}?interval=1d&range=1d"
            req = urllib.request.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
            
            with urllib.request.urlopen(req, timeout=10) as response:
                data = json.loads(response.read().decode())
                price = data['chart']['result'][0]['meta']['regularMarketPrice']
                return float(price)
        except:
            return None
    
    def fetch_crypto_price(self, ticker: str) -> Optional[float]:
        """Prix via CoinGecko"""
        try:
            # Mapping tickers communs
            crypto_ids = {
                'BTC': 'bitcoin', 'ETH': 'ethereum', 'SOL': 'solana',
                'ADA': 'cardano', 'DOT': 'polkadot', 'AVAX': 'avalanche-2',
                'MATIC': 'matic-network', 'LINK': 'chainlink', 'UNI': 'uniswap',
                'XRP': 'ripple', 'DOGE': 'dogecoin', 'SHIB': 'shiba-inu'
            }
            
            coin_id = crypto_ids.get(ticker.upper(), ticker.lower())
            url = f"https://api.coingecko.com/api/v3/simple/price?ids={coin_id}&vs_currencies=eur"
            req = urllib.request.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
            
            with urllib.request.urlopen(req, timeout=10) as response:
                data = json.loads(response.read().decode())
                return float(data[coin_id]['eur'])
        except:
            return None


# ═══════════════════════════════════════════════════════════════════════════════
# 📊 WIDGETS PERSONNALISÉS
# ═══════════════════════════════════════════════════════════════════════════════

class StatCard(QFrame):
    """Carte de statistique"""
    def __init__(self, label, value, icon="", suffix="", theme=None, accent=False, parent=None):
        super().__init__(parent)
        self.setProperty("class", "stat-card-accent" if accent else "stat-card")
        self.setMinimumHeight(100)
        
        layout = QVBoxLayout(self)
        layout.setContentsMargins(18, 14, 18, 14)
        layout.setSpacing(6)
        
        # Header
        header = QHBoxLayout()
        icon_lbl = QLabel(icon)
        icon_lbl.setStyleSheet("font-size: 18px;")
        header.addWidget(icon_lbl)
        
        label_lbl = QLabel(label)
        label_lbl.setProperty("class", "stat-label")
        header.addWidget(label_lbl)
        header.addStretch()
        
        layout.addLayout(header)
        
        # Valeur
        self.value_label = QLabel(value)
        self.value_label.setProperty("class", "stat-value")
        layout.addWidget(self.value_label)
        
        self.suffix = suffix
    
    def set_value(self, value, positive=None):
        text = f"{value}{self.suffix}" if self.suffix else value
        self.value_label.setText(text)
        
        if positive is not None:
            self.value_label.setProperty("class", "stat-value-success" if positive else "stat-value-danger")
            self.value_label.style().unpolish(self.value_label)
            self.value_label.style().polish(self.value_label)


class CategoryBadge(QLabel):
    """Badge coloré pour catégorie"""
    COLORS = {
        'Actions': '#3b82f6',
        'ETF': '#d4af37',
        'Obligations': '#10b981',
        'Crypto': '#f59e0b',
        'Immobilier': '#8b5cf6',
        'Fonds Euro': '#06b6d4',
        'Liquidités': '#6b7280',
        'SCPI': '#ec4899',
        'Autre': '#9ca3af'
    }
    
    def __init__(self, category, parent=None):
        super().__init__(category, parent)
        color = self.COLORS.get(category, '#6b7280')
        self.setStyleSheet(f"""
            background-color: {color}25;
            color: {color};
            padding: 4px 10px;
            border-radius: 6px;
            font-size: 11px;
            font-weight: 600;
        """)


class PercentBadge(QLabel):
    """Badge de pourcentage avec couleur"""
    def __init__(self, value, parent=None):
        super().__init__(parent)
        self.set_value(value)
    
    def set_value(self, value):
        color = '#10b981' if value >= 0 else '#ef4444'
        prefix = '+' if value >= 0 else ''
        self.setText(f"{prefix}{value:.2f}%")
        self.setStyleSheet(f"color: {color}; font-weight: 600; font-size: 12px;")


class MiniChart(QFrame):
    """Mini graphique sparkline"""
    def __init__(self, data: List[float], color='#d4af37', parent=None):
        super().__init__(parent)
        self.data = data
        self.color = color
        self.setMinimumSize(80, 30)
        self.setMaximumHeight(35)
    
    def paintEvent(self, event):
        super().paintEvent(event)
        if not self.data or len(self.data) < 2:
            return
        
        painter = QPainter(self)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)
        
        w, h = self.width(), self.height()
        margin = 2
        
        min_val = min(self.data)
        max_val = max(self.data)
        range_val = max_val - min_val or 1
        
        # Points
        points = []
        for i, val in enumerate(self.data):
            x = margin + (i / (len(self.data) - 1)) * (w - 2 * margin)
            y = h - margin - ((val - min_val) / range_val) * (h - 2 * margin)
            points.append((x, y))
        
        # Ligne
        pen = QPen(QColor(self.color))
        pen.setWidth(2)
        painter.setPen(pen)
        
        for i in range(len(points) - 1):
            painter.drawLine(int(points[i][0]), int(points[i][1]),
                           int(points[i+1][0]), int(points[i+1][1]))


# ═══════════════════════════════════════════════════════════════════════════════
# 📈 PAGE DASHBOARD
# ═══════════════════════════════════════════════════════════════════════════════

class DashboardPage(QWidget):
    def __init__(self, db: Database, theme: dict, parent=None):
        super().__init__(parent)
        self.db = db
        self.theme = theme
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setContentsMargins(25, 25, 25, 25)
        layout.setSpacing(20)
        
        # Header
        header = QHBoxLayout()
        title = QLabel("📊 Tableau de bord")
        title.setProperty("class", "title")
        header.addWidget(title)
        header.addStretch()
        
        # Période
        self.period_combo = QComboBox()
        self.period_combo.addItems(['30 jours', '90 jours', '6 mois', '1 an', 'Tout'])
        self.period_combo.setFixedWidth(120)
        self.period_combo.currentTextChanged.connect(self.refresh)
        header.addWidget(self.period_combo)
        
        refresh_btn = QPushButton("🔄 Actualiser les cours")
        refresh_btn.clicked.connect(self.update_prices)
        header.addWidget(refresh_btn)
        
        layout.addLayout(header)
        
        # Stats cards
        stats_grid = QHBoxLayout()
        stats_grid.setSpacing(15)
        
        self.card_value = StatCard("Patrimoine total", "0 €", "💰", "", self.theme, accent=True)
        self.card_invested = StatCard("Total investi", "0 €", "📥")
        self.card_gain = StatCard("Plus-value", "0 €", "📈")
        self.card_dividends = StatCard("Dividendes", "0 €", "💵")
        self.card_perf = StatCard("Performance", "0%", "🎯")
        
        stats_grid.addWidget(self.card_value)
        stats_grid.addWidget(self.card_invested)
        stats_grid.addWidget(self.card_gain)
        stats_grid.addWidget(self.card_dividends)
        stats_grid.addWidget(self.card_perf)
        
        layout.addLayout(stats_grid)
        
        # Charts
        charts_layout = QHBoxLayout()
        charts_layout.setSpacing(20)
        
        # Évolution
        evolution_card = QFrame()
        evolution_card.setProperty("class", "card")
        evolution_layout = QVBoxLayout(evolution_card)
        evolution_layout.setContentsMargins(15, 15, 15, 15)
        
        evo_header = QHBoxLayout()
        evo_title = QLabel("📈 Évolution du patrimoine")
        evo_title.setProperty("class", "section-title")
        evo_header.addWidget(evo_title)
        evo_header.addStretch()
        evolution_layout.addLayout(evo_header)
        
        self.line_chart = self.create_line_chart()
        self.line_view = QChartView(self.line_chart)
        self.line_view.setRenderHint(QPainter.RenderHint.Antialiasing)
        self.line_view.setMinimumHeight(250)
        evolution_layout.addWidget(self.line_view)
        
        charts_layout.addWidget(evolution_card, 2)
        
        # Répartition
        alloc_card = QFrame()
        alloc_card.setProperty("class", "card")
        alloc_layout = QVBoxLayout(alloc_card)
        alloc_layout.setContentsMargins(15, 15, 15, 15)
        
        alloc_title = QLabel("🥧 Répartition")
        alloc_title.setProperty("class", "section-title")
        alloc_layout.addWidget(alloc_title)
        
        self.pie_chart = self.create_pie_chart()
        self.pie_view = QChartView(self.pie_chart)
        self.pie_view.setRenderHint(QPainter.RenderHint.Antialiasing)
        self.pie_view.setMinimumHeight(250)
        alloc_layout.addWidget(self.pie_view)
        
        charts_layout.addWidget(alloc_card, 1)
        
        layout.addLayout(charts_layout)
        
        # Bottom row
        bottom_layout = QHBoxLayout()
        bottom_layout.setSpacing(20)
        
        # Top positions
        top_card = QFrame()
        top_card.setProperty("class", "card")
        top_layout = QVBoxLayout(top_card)
        top_layout.setContentsMargins(15, 15, 15, 15)
        
        top_title = QLabel("🏆 Top 5 positions")
        top_title.setProperty("class", "section-title")
        top_layout.addWidget(top_title)
        
        self.top_list = QTableWidget()
        self.top_list.setColumnCount(4)
        self.top_list.setHorizontalHeaderLabels(["Actif", "Valeur", "Poids", "Perf"])
        self.top_list.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeMode.Stretch)
        self.top_list.verticalHeader().setVisible(False)
        self.top_list.setMaximumHeight(200)
        top_layout.addWidget(self.top_list)
        
        bottom_layout.addWidget(top_card, 1)
        
        # Allocation vs Cible
        target_card = QFrame()
        target_card.setProperty("class", "card")
        target_layout = QVBoxLayout(target_card)
        target_layout.setContentsMargins(15, 15, 15, 15)
        
        target_title = QLabel("🎯 Allocation vs Cible")
        target_title.setProperty("class", "section-title")
        target_layout.addWidget(target_title)
        
        self.allocation_widget = QWidget()
        self.allocation_layout = QVBoxLayout(self.allocation_widget)
        self.allocation_layout.setSpacing(8)
        target_layout.addWidget(self.allocation_widget)
        
        bottom_layout.addWidget(target_card, 1)
        
        layout.addLayout(bottom_layout)
        
        # Initial refresh
        self.refresh()
    
    def create_line_chart(self):
        chart = QChart()
        chart.setBackgroundVisible(False)
        chart.legend().hide()
        chart.setAnimationOptions(QChart.AnimationOption.SeriesAnimations)
        chart.setMargins(QMargins(0, 0, 0, 0))
        return chart
    
    def create_pie_chart(self):
        chart = QChart()
        chart.setBackgroundVisible(False)
        chart.legend().setLabelColor(QColor(self.theme['text_primary']))
        chart.legend().setAlignment(Qt.AlignmentFlag.AlignRight)
        chart.setAnimationOptions(QChart.AnimationOption.SeriesAnimations)
        return chart
    
    def refresh(self):
        stats = self.db.get_portfolio_stats()
        
        # Update cards
        self.card_value.set_value(f"{stats['total_value']:,.0f} €".replace(",", " "))
        self.card_invested.set_value(f"{stats['total_invested']:,.0f} €".replace(",", " "))
        
        gain = stats['gain']
        self.card_gain.set_value(f"{'+' if gain >= 0 else ''}{gain:,.0f} €".replace(",", " "), gain >= 0)
        
        self.card_dividends.set_value(f"{stats['total_dividends']:,.0f} €".replace(",", " "))
        
        perf = stats['gain_pct']
        self.card_perf.set_value(f"{'+' if perf >= 0 else ''}{perf:.1f}%", perf >= 0)
        
        # Charts
        self.update_line_chart()
        self.update_pie_chart(stats['categories'])
        self.update_top_positions()
        self.update_allocation_comparison()
        
        # Save snapshot
        if stats['total_value'] > 0:
            self.db.save_snapshot(stats['total_value'], stats['total_invested'], stats['total_dividends'])
    
    def update_line_chart(self):
        self.line_chart.removeAllSeries()
        
        # Période
        period_map = {'30 jours': 30, '90 jours': 90, '6 mois': 180, '1 an': 365, 'Tout': 3650}
        days = period_map.get(self.period_combo.currentText(), 180)
        
        history = self.db.get_history(days)
        if len(history) < 2:
            return
        
        # Série principale
        series = QLineSeries()
        series.setColor(QColor(self.theme['accent']))
        pen = QPen(QColor(self.theme['accent']))
        pen.setWidth(2)
        series.setPen(pen)
        
        # Area sous la courbe
        area_series = QAreaSeries(series)
        gradient = QLinearGradient(0, 0, 0, 1)
        gradient.setColorAt(0, QColor(self.theme['accent'] + '40'))
        gradient.setColorAt(1, QColor(self.theme['accent'] + '00'))
        gradient.setCoordinateMode(QLinearGradient.CoordinateMode.ObjectBoundingMode)
        area_series.setBrush(QBrush(gradient))
        area_series.setPen(QPen(Qt.PenStyle.NoPen))
        
        for h in history:
            date = datetime.strptime(h['date'], "%Y-%m-%d")
            series.append(date.timestamp() * 1000, h['total_value'])
        
        self.line_chart.addSeries(area_series)
        self.line_chart.addSeries(series)
        
        # Axes
        axis_x = QDateTimeAxis()
        axis_x.setFormat("dd/MM")
        axis_x.setLabelsColor(QColor(self.theme['text_secondary']))
        axis_x.setGridLineColor(QColor(self.theme['border']))
        
        axis_y = QValueAxis()
        axis_y.setLabelsColor(QColor(self.theme['text_secondary']))
        axis_y.setGridLineColor(QColor(self.theme['border']))
        
        self.line_chart.addAxis(axis_x, Qt.AlignmentFlag.AlignBottom)
        self.line_chart.addAxis(axis_y, Qt.AlignmentFlag.AlignLeft)
        
        series.attachAxis(axis_x)
        series.attachAxis(axis_y)
        area_series.attachAxis(axis_x)
        area_series.attachAxis(axis_y)
    
    def update_pie_chart(self, categories):
        self.pie_chart.removeAllSeries()
        if not categories:
            return
        
        series = QPieSeries()
        colors = CategoryBadge.COLORS
        
        for cat, value in sorted(categories.items(), key=lambda x: -x[1]):
            slice_ = series.append(f"{cat}", value)
            slice_.setColor(QColor(colors.get(cat, '#6b7280')))
            slice_.setLabelVisible(True)
            slice_.setLabelColor(QColor(self.theme['text_primary']))
            slice_.setLabel(f"{cat}: {value:,.0f}€")
        
        self.pie_chart.addSeries(series)
    
    def update_top_positions(self):
        assets = self.db.get_assets()
        total = sum(a['quantity'] * (a['current_price'] or a['buy_price']) for a in assets)
        
        sorted_assets = sorted(assets, key=lambda a: a['quantity'] * (a['current_price'] or a['buy_price']), reverse=True)[:5]
        
        self.top_list.setRowCount(len(sorted_assets))
        for i, a in enumerate(sorted_assets):
            value = a['quantity'] * (a['current_price'] or a['buy_price'])
            invested = a['quantity'] * a['buy_price']
            perf = ((value - invested) / invested * 100) if invested > 0 else 0
            weight = (value / total * 100) if total > 0 else 0
            
            self.top_list.setItem(i, 0, QTableWidgetItem(a['name']))
            self.top_list.setItem(i, 1, QTableWidgetItem(f"{value:,.0f} €".replace(",", " ")))
            self.top_list.setItem(i, 2, QTableWidgetItem(f"{weight:.1f}%"))
            
            perf_item = QTableWidgetItem(f"{'+' if perf >= 0 else ''}{perf:.1f}%")
            perf_item.setForeground(QColor(self.theme['success'] if perf >= 0 else self.theme['danger']))
            self.top_list.setItem(i, 3, perf_item)
    
    def update_allocation_comparison(self):
        # Clear
        while self.allocation_layout.count():
            child = self.allocation_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        stats = self.db.get_portfolio_stats()
        targets = self.db.get_target_allocation()
        total = stats['total_value'] or 1
        
        for target in targets:
            cat = target['category']
            target_pct = target['target_percent']
            actual_pct = (stats['categories'].get(cat, 0) / total * 100)
            diff = actual_pct - target_pct
            
            row = QWidget()
            row_layout = QHBoxLayout(row)
            row_layout.setContentsMargins(0, 2, 0, 2)
            
            # Catégorie
            cat_lbl = QLabel(cat)
            cat_lbl.setFixedWidth(80)
            cat_lbl.setStyleSheet(f"font-size: 12px;")
            row_layout.addWidget(cat_lbl)
            
            # Barre de progression
            bar = QProgressBar()
            bar.setMaximum(100)
            bar.setValue(int(min(actual_pct, 100)))
            bar.setFormat(f"{actual_pct:.0f}% / {target_pct:.0f}%")
            bar.setFixedHeight(18)
            row_layout.addWidget(bar)
            
            # Différence
            diff_color = self.theme['success'] if abs(diff) < 5 else (self.theme['warning'] if abs(diff) < 10 else self.theme['danger'])
            diff_lbl = QLabel(f"{'+' if diff > 0 else ''}{diff:.0f}%")
            diff_lbl.setStyleSheet(f"color: {diff_color}; font-size: 11px; font-weight: 600;")
            diff_lbl.setFixedWidth(45)
            row_layout.addWidget(diff_lbl)
            
            self.allocation_layout.addWidget(row)
    
    def update_prices(self):
        """Lance la mise à jour des prix en arrière-plan"""
        assets = self.db.get_assets()
        if not assets:
            return
        
        self.price_thread = PriceUpdateThread(assets)
        self.price_thread.price_updated.connect(self.on_price_updated)
        self.price_thread.update_complete.connect(self.refresh)
        self.price_thread.start()
    
    def on_price_updated(self, asset_id, price):
        self.db.update_asset_price(asset_id, price)


# ═══════════════════════════════════════════════════════════════════════════════
# 💼 PAGE PORTEFEUILLE
# ═══════════════════════════════════════════════════════════════════════════════

class PortfolioPage(QWidget):
    def __init__(self, db: Database, theme: dict, parent=None):
        super().__init__(parent)
        self.db = db
        self.theme = theme
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setContentsMargins(25, 25, 25, 25)
        layout.setSpacing(15)
        
        # Header
        header = QHBoxLayout()
        title = QLabel("💼 Mon portefeuille")
        title.setProperty("class", "title")
        header.addWidget(title)
        header.addStretch()
        
        # Filtres
        self.category_filter = QComboBox()
        self.category_filter.addItems(['Toutes', 'Actions', 'ETF', 'Obligations', 'Crypto', 'Immobilier', 'Liquidités'])
        self.category_filter.currentTextChanged.connect(self.refresh)
        header.addWidget(self.category_filter)
        
        self.search_input = QLineEdit()
        self.search_input.setPlaceholderText("🔍 Rechercher...")
        self.search_input.setFixedWidth(200)
        self.search_input.textChanged.connect(self.refresh)
        header.addWidget(self.search_input)
        
        add_btn = QPushButton("+ Ajouter")
        add_btn.setProperty("class", "primary")
        add_btn.clicked.connect(self.add_asset)
        header.addWidget(add_btn)
        
        import_btn = QPushButton("📥 Import")
        import_btn.clicked.connect(self.import_csv)
        header.addWidget(import_btn)
        
        layout.addLayout(header)
        
        # Table
        self.table = QTableWidget()
        self.table.setColumnCount(10)
        self.table.setHorizontalHeaderLabels([
            "Actif", "Ticker", "Catégorie", "Qté", "PRU", "Cours", "Valeur", "+/-", "%", "Actions"
        ])
        self.table.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeMode.Stretch)
        self.table.verticalHeader().setVisible(False)
        self.table.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.table.customContextMenuRequested.connect(self.show_context_menu)
        
        layout.addWidget(self.table)
        
        # Summary
        summary = QFrame()
        summary.setProperty("class", "card")
        summary.setMaximumHeight(50)
        summary_layout = QHBoxLayout(summary)
        summary_layout.setContentsMargins(15, 10, 15, 10)
        
        self.summary_label = QLabel("0 actifs")
        self.summary_label.setProperty("class", "small")
        summary_layout.addWidget(self.summary_label)
        
        summary_layout.addStretch()
        
        export_btn = QPushButton("📤 Exporter CSV")
        export_btn.setProperty("class", "ghost")
        export_btn.clicked.connect(self.export_csv)
        summary_layout.addWidget(export_btn)
        
        layout.addWidget(summary)
        
        self.refresh()
    
    def refresh(self):
        assets = self.db.get_assets()
        
        # Filtres
        cat_filter = self.category_filter.currentText()
        search = self.search_input.text().lower()
        
        if cat_filter != 'Toutes':
            assets = [a for a in assets if a['category'] == cat_filter]
        if search:
            assets = [a for a in assets if search in a['name'].lower() or search in (a.get('ticker') or '').lower()]
        
        self.table.setRowCount(len(assets))
        total_value = 0
        total_invested = 0
        
        for i, a in enumerate(assets):
            value = a['quantity'] * (a['current_price'] or a['buy_price'])
            invested = a['quantity'] * a['buy_price']
            gain = value - invested
            perf = (gain / invested * 100) if invested > 0 else 0
            
            total_value += value
            total_invested += invested
            
            self.table.setItem(i, 0, QTableWidgetItem(a['name']))
            self.table.setItem(i, 1, QTableWidgetItem(a.get('ticker') or '-'))
            
            # Badge catégorie
            cat_widget = QWidget()
            cat_layout = QHBoxLayout(cat_widget)
            cat_layout.setContentsMargins(5, 2, 5, 2)
            cat_layout.addWidget(CategoryBadge(a['category']))
            cat_layout.addStretch()
            self.table.setCellWidget(i, 2, cat_widget)
            
            self.table.setItem(i, 3, QTableWidgetItem(f"{a['quantity']:.4f}".rstrip('0').rstrip('.')))
            self.table.setItem(i, 4, QTableWidgetItem(f"{a['buy_price']:.2f}"))
            self.table.setItem(i, 5, QTableWidgetItem(f"{a['current_price']:.2f}" if a['current_price'] else '-'))
            self.table.setItem(i, 6, QTableWidgetItem(f"{value:,.0f}".replace(",", " ")))
            
            gain_item = QTableWidgetItem(f"{'+' if gain >= 0 else ''}{gain:,.0f}".replace(",", " "))
            gain_item.setForeground(QColor(self.theme['success'] if gain >= 0 else self.theme['danger']))
            self.table.setItem(i, 7, gain_item)
            
            perf_item = QTableWidgetItem(f"{'+' if perf >= 0 else ''}{perf:.1f}%")
            perf_item.setForeground(QColor(self.theme['success'] if perf >= 0 else self.theme['danger']))
            self.table.setItem(i, 8, perf_item)
            
            # Actions
            actions = QWidget()
            actions_layout = QHBoxLayout(actions)
            actions_layout.setContentsMargins(2, 2, 2, 2)
            actions_layout.setSpacing(2)
            
            edit_btn = QPushButton("✏️")
            edit_btn.setProperty("class", "icon-btn")
            edit_btn.setFixedSize(28, 28)
            edit_btn.clicked.connect(lambda _, asset=a: self.edit_asset(asset))
            actions_layout.addWidget(edit_btn)
            
            del_btn = QPushButton("🗑️")
            del_btn.setProperty("class", "icon-btn")
            del_btn.setFixedSize(28, 28)
            del_btn.clicked.connect(lambda _, asset=a: self.delete_asset(asset))
            actions_layout.addWidget(del_btn)
            
            self.table.setCellWidget(i, 9, actions)
        
        # Summary
        total_gain = total_value - total_invested
        self.summary_label.setText(
            f"{len(assets)} actifs • Valeur: {total_value:,.0f}€ • Investi: {total_invested:,.0f}€ • P/L: {'+' if total_gain >= 0 else ''}{total_gain:,.0f}€".replace(",", " ")
        )
    
    def add_asset(self):
        dialog = AssetDialog(self.db, self.theme, parent=self)
        if dialog.exec():
            self.refresh()
    
    def edit_asset(self, asset):
        dialog = AssetDialog(self.db, self.theme, asset=asset, parent=self)
        if dialog.exec():
            self.refresh()
    
    def delete_asset(self, asset):
        reply = QMessageBox.question(self, "Supprimer", f"Supprimer '{asset['name']}' ?")
        if reply == QMessageBox.StandardButton.Yes:
            self.db.delete_asset(asset['id'])
            self.refresh()
    
    def show_context_menu(self, pos):
        row = self.table.rowAt(pos.y())
        if row < 0:
            return
        
        menu = QMenu(self)
        menu.addAction("✏️ Modifier", lambda: self.edit_asset(self.get_asset_at_row(row)))
        menu.addAction("📊 Transactions", lambda: self.show_transactions(self.get_asset_at_row(row)))
        menu.addAction("💰 Ajouter dividende", lambda: self.add_dividend(self.get_asset_at_row(row)))
        menu.addSeparator()
        menu.addAction("🗑️ Supprimer", lambda: self.delete_asset(self.get_asset_at_row(row)))
        menu.exec(self.table.mapToGlobal(pos))
    
    def get_asset_at_row(self, row):
        assets = self.db.get_assets()
        cat_filter = self.category_filter.currentText()
        search = self.search_input.text().lower()
        
        if cat_filter != 'Toutes':
            assets = [a for a in assets if a['category'] == cat_filter]
        if search:
            assets = [a for a in assets if search in a['name'].lower()]
        
        return assets[row] if row < len(assets) else None
    
    def show_transactions(self, asset):
        if asset:
            QMessageBox.information(self, "Transactions", f"Transactions pour {asset['name']}\n\n(Fonctionnalité à développer)")
    
    def add_dividend(self, asset):
        if asset:
            amount, ok = QDoubleSpinBox(), False
            # Simplified - would use a dialog in production
            QMessageBox.information(self, "Dividende", f"Ajouter un dividende pour {asset['name']}\n\n(Fonctionnalité à développer)")
    
    def import_csv(self):
        file_path, _ = QFileDialog.getOpenFileName(self, "Importer CSV", "", "CSV (*.csv)")
        if file_path:
            try:
                with open(file_path, 'r', encoding='utf-8') as f:
                    reader = csv.DictReader(f, delimiter=';')
                    count = 0
                    for row in reader:
                        name = row.get('Nom', row.get('name', row.get('LIBELLE', '')))
                        ticker = row.get('Ticker', row.get('ticker', row.get('ISIN', '')))
                        quantity = float(row.get('Quantité', row.get('quantity', 0)) or 0)
                        price = float(row.get('PRU', row.get('buy_price', 0)) or 0)
                        
                        if name and quantity > 0:
                            self.db.add_asset(name=name, ticker=ticker, category='Autre', quantity=quantity, buy_price=price)
                            count += 1
                    
                    QMessageBox.information(self, "Import", f"{count} actifs importés !")
                    self.refresh()
            except Exception as e:
                QMessageBox.warning(self, "Erreur", str(e))
    
    def export_csv(self):
        file_path, _ = QFileDialog.getSaveFileName(self, "Exporter CSV", "portfolio.csv", "CSV (*.csv)")
        if file_path:
            assets = self.db.get_assets()
            with open(file_path, 'w', newline='', encoding='utf-8') as f:
                writer = csv.DictWriter(f, fieldnames=['name', 'ticker', 'category', 'quantity', 'buy_price', 'current_price'])
                writer.writeheader()
                for a in assets:
                    writer.writerow({k: a.get(k, '') for k in writer.fieldnames})
            QMessageBox.information(self, "Export", f"Exporté vers {file_path}")


# ═══════════════════════════════════════════════════════════════════════════════
# ➕ DIALOG ASSET
# ═══════════════════════════════════════════════════════════════════════════════

class AssetDialog(QDialog):
    CATEGORIES = ['Actions', 'ETF', 'Obligations', 'Crypto', 'Immobilier', 'Fonds Euro', 'SCPI', 'Liquidités', 'Autre']
    
    def __init__(self, db: Database, theme: dict, asset=None, parent=None):
        super().__init__(parent)
        self.db = db
        self.theme = theme
        self.asset = asset
        self.setWindowTitle("Modifier" if asset else "Ajouter un actif")
        self.setMinimumWidth(480)
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setSpacing(15)
        layout.setContentsMargins(25, 25, 25, 25)
        
        # Form
        form = QFormLayout()
        form.setSpacing(12)
        
        self.name_input = QLineEdit()
        self.name_input.setPlaceholderText("Ex: Apple Inc., ETF World...")
        form.addRow("Nom *", self.name_input)
        
        ticker_row = QHBoxLayout()
        self.ticker_input = QLineEdit()
        self.ticker_input.setPlaceholderText("AAPL, CW8, BTC...")
        ticker_row.addWidget(self.ticker_input)
        self.isin_input = QLineEdit()
        self.isin_input.setPlaceholderText("ISIN (optionnel)")
        self.isin_input.setFixedWidth(150)
        ticker_row.addWidget(self.isin_input)
        form.addRow("Ticker / ISIN", ticker_row)
        
        self.category_combo = QComboBox()
        self.category_combo.addItems(self.CATEGORIES)
        form.addRow("Catégorie *", self.category_combo)
        
        qty_row = QHBoxLayout()
        self.quantity_input = QDoubleSpinBox()
        self.quantity_input.setRange(0, 999999999)
        self.quantity_input.setDecimals(6)
        qty_row.addWidget(self.quantity_input)
        self.buy_price_input = QDoubleSpinBox()
        self.buy_price_input.setRange(0, 999999999)
        self.buy_price_input.setDecimals(4)
        self.buy_price_input.setSuffix(" €")
        qty_row.addWidget(QLabel("×"))
        qty_row.addWidget(self.buy_price_input)
        form.addRow("Quantité × PRU *", qty_row)
        
        self.current_price_input = QDoubleSpinBox()
        self.current_price_input.setRange(0, 999999999)
        self.current_price_input.setDecimals(4)
        self.current_price_input.setSuffix(" €")
        form.addRow("Cours actuel", self.current_price_input)
        
        broker_row = QHBoxLayout()
        self.account_input = QLineEdit()
        self.account_input.setPlaceholderText("PEA, CTO, AV...")
        broker_row.addWidget(self.account_input)
        self.broker_input = QLineEdit()
        self.broker_input.setPlaceholderText("Courtier")
        broker_row.addWidget(self.broker_input)
        form.addRow("Compte / Courtier", broker_row)
        
        self.auto_update_check = QCheckBox("Actualisation automatique des cours")
        self.auto_update_check.setChecked(True)
        form.addRow("", self.auto_update_check)
        
        layout.addLayout(form)
        
        # Buttons
        buttons = QHBoxLayout()
        buttons.addStretch()
        
        cancel_btn = QPushButton("Annuler")
        cancel_btn.clicked.connect(self.reject)
        buttons.addWidget(cancel_btn)
        
        save_btn = QPushButton("💾 Enregistrer")
        save_btn.setProperty("class", "primary")
        save_btn.clicked.connect(self.save)
        buttons.addWidget(save_btn)
        
        layout.addLayout(buttons)
        
        # Load existing
        if self.asset:
            self.name_input.setText(self.asset['name'])
            self.ticker_input.setText(self.asset.get('ticker') or '')
            self.isin_input.setText(self.asset.get('isin') or '')
            self.category_combo.setCurrentText(self.asset['category'])
            self.quantity_input.setValue(self.asset['quantity'])
            self.buy_price_input.setValue(self.asset['buy_price'])
            self.current_price_input.setValue(self.asset.get('current_price') or self.asset['buy_price'])
            self.account_input.setText(self.asset.get('account') or '')
            self.broker_input.setText(self.asset.get('broker') or '')
            self.auto_update_check.setChecked(bool(self.asset.get('auto_update', 1)))
    
    def save(self):
        name = self.name_input.text().strip()
        if not name:
            QMessageBox.warning(self, "Erreur", "Le nom est obligatoire")
            return
        
        data = {
            'name': name,
            'ticker': self.ticker_input.text().strip() or None,
            'isin': self.isin_input.text().strip() or None,
            'category': self.category_combo.currentText(),
            'quantity': self.quantity_input.value(),
            'buy_price': self.buy_price_input.value(),
            'current_price': self.current_price_input.value() or self.buy_price_input.value(),
            'account': self.account_input.text().strip() or None,
            'broker': self.broker_input.text().strip() or None,
            'auto_update': 1 if self.auto_update_check.isChecked() else 0
        }
        
        if self.asset:
            self.db.update_asset(self.asset['id'], **data)
        else:
            self.db.add_asset(**data)
        
        self.accept()


# ═══════════════════════════════════════════════════════════════════════════════
# 🎯 PAGE OBJECTIFS
# ═══════════════════════════════════════════════════════════════════════════════

class GoalsPage(QWidget):
    def __init__(self, db: Database, theme: dict, parent=None):
        super().__init__(parent)
        self.db = db
        self.theme = theme
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setContentsMargins(25, 25, 25, 25)
        layout.setSpacing(20)
        
        # Header
        header = QHBoxLayout()
        title = QLabel("🎯 Mes objectifs")
        title.setProperty("class", "title")
        header.addWidget(title)
        header.addStretch()
        
        add_btn = QPushButton("+ Nouvel objectif")
        add_btn.setProperty("class", "primary")
        add_btn.clicked.connect(self.add_goal)
        header.addWidget(add_btn)
        
        layout.addLayout(header)
        
        # Scroll area
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setFrameShape(QFrame.Shape.NoFrame)
        
        self.goals_container = QWidget()
        self.goals_layout = QVBoxLayout(self.goals_container)
        self.goals_layout.setSpacing(15)
        self.goals_layout.addStretch()
        
        scroll.setWidget(self.goals_container)
        layout.addWidget(scroll)
        
        self.refresh()
    
    def refresh(self):
        # Clear
        while self.goals_layout.count() > 1:
            item = self.goals_layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        
        goals = self.db.get_goals()
        stats = self.db.get_portfolio_stats()
        current = stats['total_value']
        
        for goal in goals:
            card = self.create_goal_card(goal, current)
            self.goals_layout.insertWidget(self.goals_layout.count() - 1, card)
    
    def create_goal_card(self, goal, current):
        card = QFrame()
        card.setProperty("class", "card")
        layout = QVBoxLayout(card)
        layout.setContentsMargins(20, 15, 20, 15)
        
        # Header
        header = QHBoxLayout()
        name = QLabel(f"🎯 {goal['name']}")
        name.setProperty("class", "section-title")
        header.addWidget(name)
        header.addStretch()
        
        del_btn = QPushButton("🗑️")
        del_btn.setProperty("class", "icon-btn")
        del_btn.setFixedSize(28, 28)
        del_btn.clicked.connect(lambda: self.delete_goal(goal['id']))
        header.addWidget(del_btn)
        
        layout.addLayout(header)
        
        # Progress
        progress = min(100, (current / goal['target_amount'] * 100)) if goal['target_amount'] > 0 else 0
        
        bar = QProgressBar()
        bar.setValue(int(progress))
        bar.setFormat(f"{progress:.1f}%")
        layout.addWidget(bar)
        
        # Stats
        stats_row = QHBoxLayout()
        stats_row.addWidget(QLabel(f"Actuel: {current:,.0f}€".replace(",", " ")))
        stats_row.addStretch()
        stats_row.addWidget(QLabel(f"Objectif: {goal['target_amount']:,.0f}€".replace(",", " ")))
        
        remaining = goal['target_amount'] - current
        if remaining > 0:
            stats_row.addWidget(QLabel(f" • Reste: {remaining:,.0f}€".replace(",", " ")))
        
        layout.addLayout(stats_row)
        
        return card
    
    def add_goal(self):
        dialog = GoalDialog(self.db, self.theme, parent=self)
        if dialog.exec():
            self.refresh()
    
    def delete_goal(self, goal_id):
        reply = QMessageBox.question(self, "Supprimer", "Supprimer cet objectif ?")
        if reply == QMessageBox.StandardButton.Yes:
            self.db.delete_goal(goal_id)
            self.refresh()


class GoalDialog(QDialog):
    def __init__(self, db: Database, theme: dict, parent=None):
        super().__init__(parent)
        self.db = db
        self.setWindowTitle("Nouvel objectif")
        self.setMinimumWidth(400)
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setSpacing(15)
        layout.setContentsMargins(25, 25, 25, 25)
        
        form = QFormLayout()
        form.setSpacing(12)
        
        self.name_input = QLineEdit()
        self.name_input.setPlaceholderText("Ex: Apport immobilier")
        form.addRow("Nom *", self.name_input)
        
        self.target_input = QDoubleSpinBox()
        self.target_input.setRange(0, 999999999)
        self.target_input.setValue(50000)
        self.target_input.setSuffix(" €")
        form.addRow("Montant cible *", self.target_input)
        
        self.date_input = QDateEdit()
        self.date_input.setCalendarPopup(True)
        self.date_input.setDate(QDate.currentDate().addYears(5))
        form.addRow("Date cible", self.date_input)
        
        layout.addLayout(form)
        
        # Buttons
        buttons = QHBoxLayout()
        buttons.addStretch()
        cancel_btn = QPushButton("Annuler")
        cancel_btn.clicked.connect(self.reject)
        buttons.addWidget(cancel_btn)
        save_btn = QPushButton("💾 Créer")
        save_btn.setProperty("class", "primary")
        save_btn.clicked.connect(self.save)
        buttons.addWidget(save_btn)
        layout.addLayout(buttons)
    
    def save(self):
        name = self.name_input.text().strip()
        if not name:
            return
        self.db.add_goal(name, self.target_input.value(), self.date_input.date().toString("yyyy-MM-dd"))
        self.accept()


# ═══════════════════════════════════════════════════════════════════════════════
# ⚙️ PAGE PARAMÈTRES
# ═══════════════════════════════════════════════════════════════════════════════

class SettingsPage(QWidget):
    theme_changed = pyqtSignal(str)
    
    def __init__(self, db: Database, theme: dict, parent=None):
        super().__init__(parent)
        self.db = db
        self.theme = theme
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setContentsMargins(25, 25, 25, 25)
        layout.setSpacing(20)
        
        title = QLabel("⚙️ Paramètres")
        title.setProperty("class", "title")
        layout.addWidget(title)
        
        # Apparence
        appearance_group = QGroupBox("🎨 Apparence")
        appearance_layout = QVBoxLayout(appearance_group)
        
        theme_row = QHBoxLayout()
        theme_row.addWidget(QLabel("Thème :"))
        self.theme_combo = QComboBox()
        self.theme_combo.addItems(['Sombre', 'Clair'])
        self.theme_combo.currentTextChanged.connect(self.on_theme_change)
        theme_row.addWidget(self.theme_combo)
        theme_row.addStretch()
        appearance_layout.addLayout(theme_row)
        
        layout.addWidget(appearance_group)
        
        # Données
        data_group = QGroupBox("💾 Données")
        data_layout = QVBoxLayout(data_group)
        
        db_path = QLabel(f"📁 Base: {self.db.db_path}")
        db_path.setProperty("class", "small")
        data_layout.addWidget(db_path)
        
        backup_row = QHBoxLayout()
        backup_btn = QPushButton("📤 Exporter sauvegarde")
        backup_btn.clicked.connect(self.export_backup)
        backup_row.addWidget(backup_btn)
        
        restore_btn = QPushButton("📥 Restaurer sauvegarde")
        restore_btn.clicked.connect(self.import_backup)
        backup_row.addWidget(restore_btn)
        backup_row.addStretch()
        data_layout.addLayout(backup_row)
        
        reset_btn = QPushButton("⚠️ Réinitialiser les données")
        reset_btn.setProperty("class", "danger")
        reset_btn.clicked.connect(self.reset_data)
        data_layout.addWidget(reset_btn)
        
        layout.addWidget(data_group)
        
        # À propos
        about_group = QGroupBox("ℹ️ À propos")
        about_layout = QVBoxLayout(about_group)
        about_layout.addWidget(QLabel("<b>ScoreInvest Portfolio Tracker</b>"))
        about_layout.addWidget(QLabel("Version 2.0 - Février 2026"))
        about_layout.addWidget(QLabel(""))
        about_layout.addWidget(QLabel("🔒 100% local - Vos données restent privées"))
        about_layout.addWidget(QLabel(""))
        
        site_btn = QPushButton("🌐 Visiter ScoreInvest.fr")
        site_btn.setProperty("class", "primary")
        site_btn.clicked.connect(lambda: __import__('webbrowser').open('https://scoreinvest.fr'))
        about_layout.addWidget(site_btn)
        
        layout.addWidget(about_group)
        layout.addStretch()
    
    def on_theme_change(self, theme_name):
        theme_key = 'dark' if theme_name == 'Sombre' else 'light'
        self.db.set_setting('theme', theme_key)
        self.theme_changed.emit(theme_key)
    
    def export_backup(self):
        path, _ = QFileDialog.getSaveFileName(self, "Exporter", "scoreinvest_backup.db", "Database (*.db)")
        if path:
            self.db.export_backup(path)
            QMessageBox.information(self, "Succès", f"Sauvegarde exportée vers:\n{path}")
    
    def import_backup(self):
        path, _ = QFileDialog.getOpenFileName(self, "Restaurer", "", "Database (*.db)")
        if path:
            reply = QMessageBox.warning(self, "Attention", "Ceci remplacera toutes vos données actuelles.\nContinuer ?",
                                       QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
            if reply == QMessageBox.StandardButton.Yes:
                self.db.import_backup(path)
                QMessageBox.information(self, "Succès", "Données restaurées !")
    
    def reset_data(self):
        reply = QMessageBox.warning(self, "⚠️ Attention", "Supprimer TOUTES les données ?",
                                   QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        if reply == QMessageBox.StandardButton.Yes:
            cursor = self.db.conn.cursor()
            cursor.execute("DELETE FROM assets")
            cursor.execute("DELETE FROM transactions")
            cursor.execute("DELETE FROM dividends")
            cursor.execute("DELETE FROM portfolio_history")
            cursor.execute("DELETE FROM goals")
            self.db.conn.commit()
            QMessageBox.information(self, "Succès", "Données supprimées.")


# ═══════════════════════════════════════════════════════════════════════════════
# 🏠 FENÊTRE PRINCIPALE
# ═══════════════════════════════════════════════════════════════════════════════

class MainWindow(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("ScoreInvest Portfolio Tracker v2.0")
        self.setMinimumSize(1280, 800)
        
        # Database
        self.db = Database()
        
        # Theme
        theme_key = self.db.get_setting('theme', 'dark')
        self.current_theme = THEMES.get(theme_key, THEMES['dark'])
        
        self.setup_ui()
        self.apply_theme(theme_key)
    
    def setup_ui(self):
        central = QWidget()
        self.setCentralWidget(central)
        
        main_layout = QHBoxLayout(central)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)
        
        # Sidebar
        sidebar = QFrame()
        sidebar.setObjectName("sidebar")
        sidebar.setFixedWidth(220)
        sidebar_layout = QVBoxLayout(sidebar)
        sidebar_layout.setContentsMargins(0, 0, 0, 0)
        sidebar_layout.setSpacing(0)
        
        # Logo
        logo = QLabel("💰 ScoreInvest")
        logo.setObjectName("logo")
        sidebar_layout.addWidget(logo)
        
        version = QLabel("Portfolio Tracker v2.0")
        version.setStyleSheet(f"color: {self.current_theme['text_muted']}; font-size: 11px; padding: 0 18px 15px;")
        sidebar_layout.addWidget(version)
        
        # Navigation
        self.nav_buttons = []
        nav_items = [
            ("📊", "Tableau de bord", 0),
            ("💼", "Portefeuille", 1),
            ("🎯", "Objectifs", 2),
            ("⚙️", "Paramètres", 3),
        ]
        
        for icon, text, index in nav_items:
            btn = QPushButton(f"  {icon}  {text}")
            btn.setCheckable(True)
            btn.clicked.connect(lambda _, i=index: self.switch_page(i))
            sidebar_layout.addWidget(btn)
            self.nav_buttons.append(btn)
        
        sidebar_layout.addStretch()
        
        # Status
        self.status_label = QLabel("🟢 Prêt")
        self.status_label.setStyleSheet(f"color: {self.current_theme['text_muted']}; font-size: 11px; padding: 15px 18px;")
        sidebar_layout.addWidget(self.status_label)
        
        main_layout.addWidget(sidebar)
        
        # Pages
        self.stack = QStackedWidget()
        
        self.dashboard_page = DashboardPage(self.db, self.current_theme)
        self.portfolio_page = PortfolioPage(self.db, self.current_theme)
        self.goals_page = GoalsPage(self.db, self.current_theme)
        self.settings_page = SettingsPage(self.db, self.current_theme)
        
        self.settings_page.theme_changed.connect(self.apply_theme)
        
        self.stack.addWidget(self.dashboard_page)
        self.stack.addWidget(self.portfolio_page)
        self.stack.addWidget(self.goals_page)
        self.stack.addWidget(self.settings_page)
        
        main_layout.addWidget(self.stack)
        
        self.switch_page(0)
    
    def switch_page(self, index):
        self.stack.setCurrentIndex(index)
        
        for i, btn in enumerate(self.nav_buttons):
            btn.setChecked(i == index)
            btn.setProperty("active", "true" if i == index else "false")
            btn.style().unpolish(btn)
            btn.style().polish(btn)
        
        # Refresh
        if index == 0:
            self.dashboard_page.refresh()
        elif index == 1:
            self.portfolio_page.refresh()
        elif index == 2:
            self.goals_page.refresh()
    
    def apply_theme(self, theme_key):
        self.current_theme = THEMES.get(theme_key, THEMES['dark'])
        QApplication.instance().setStyleSheet(get_stylesheet(self.current_theme))


# ═══════════════════════════════════════════════════════════════════════════════
# 🚀 MAIN
# ═══════════════════════════════════════════════════════════════════════════════

def main():
    app = QApplication(sys.argv)
    app.setStyle("Fusion")
    
    # Default dark theme
    app.setStyleSheet(get_stylesheet(THEMES['dark']))
    
    window = MainWindow()
    window.show()
    
    sys.exit(app.exec())


if __name__ == "__main__":
    main()
